#
# Filename: upgrade.sh
# 
# (C) Copyright 2017 Silver Peak Systems, Inc.  
# All rights reserved.
#
# Helps writeimage.sh install images from the future. This script must be
# packaged in an image zip, and should never get installed into an image
# location on disk.
#
# On Yocto (and future platforms):
#
#    writeimage.sh expects this file inside the image zip and will source
#    it as soon as the image download has completed. At this point it will
#    run upgrade_init() - reserved for future use. Throughout the install
#    process, it will run the other hooks - compatibilty checks etc.
#
# On FC6 and earlier platforms:
#
#    writeimage.sh does not look for this file in the image zip. We use
#    the build_version.sh in the zip (not the inner tarball) to trick the
#    old writeimage.sh into sourcing this file as it tries to "slurp in the
#    image's build version information". At that point, it will run
#    upgrade_fc6_writeimage_hooks, which calls upgrade_init() and installs
#    the other hooks by rewriting the writeimage_graft_X functions.
#

# Determine which platform we're running
PLATFORM_VERSION=0  # FC6 and earlier
if [ -f /etc/os-release ]; then
    PLATFORM_VERSION=1  # Yocto
fi

# -----------------------------------------------------------------------------
# Init
#
# Reserved for future use.
# 
upgrade_init()
{
    #Fix USE_SYSTEMD_BOOT for secure boot boxes.
    # Only manufacture.sh flow sets CFG_MODEL so support scripts
    # writeimage.sh called got wrong USE_SYSTEMD_BOOT setting
    # on regular upgrade.
    # Let's fix that up here for versions where writeimage.sh
    # does not have fix incorporated in it.
    SYSTEMD_BOOT_MODELS=(
        "EC10108"
        "EC10150"
        "EC10170"
    )
    for model in "${SYSTEMD_BOOT_MODELS[@]}"; do
        if [ "${MODEL}" == "$model" ]; then
            USE_SYSTEMD_BOOT=1
            break
        fi
    done
    
    return;
}

# -----------------------------------------------------------------------------
# Check compatibility
# 
# Verify that the running platform, hardware, and software support upgrading to
# this image version.
# 
# Return code indicates the compatibility:
# See md_image.c
#
COMPAT_SUCCESS=0
COMPAT_FAIL_PLATFORM=10
COMPAT_FAIL_MODEL=11
COMPAT_FAIL_BASEBOARD=12
# In writeimage.sh
COMPAT_FAIL_SOFTWARE=13
#
# The codes above are in sync with mgmtd, which will map them to friendly
# message for the user if the compatibility check fails.
#
HAVE_UPGRADE_CHECK_COMPATIBILITY=y
upgrade_check_compatibility()
{
    if [ ! -z "$1" ]; then
        CURRENT_MODEL=$1
    fi

    if [ $PLATFORM_VERSION -eq 0 ]; then
        . /sbin/product.sh

        if [ -z "${CURRENT_MODEL}" ]; then
            set_cur_model
            CURRENT_MODEL="${MODEL}"
        fi

        set_bb_cur_type

        # Some of the pre-Yocto definitions were capitalized, e.g. "Azure".
        # Need to make these uppercase
        CURRENT_BASEBOARD=`echo ${BB_PRODUCT} | awk '{ print toupper($0) }'`

    elif [ $PLATFORM_VERSION -eq 1 ]; then
        . /sbin/product.sh

        if [ -z "${CURRENT_MODEL}" ]; then
            set_cur_model
            CURRENT_MODEL="${MODEL}"
        fi

        . /sbin/baseboard.sh
        set_baseboard
        CURRENT_BASEBOARD="${BASEBOARD}"
    else
        echo "Platform not recognized"
        return $COMPAT_FAIL_PLATFORM
    fi

    # Models supported by this image
    SUPPORTED_MODELS=(
        "NX700"
        "NX1700"
        "NX2700"
        "NX3700"
        "NX5700"
        "NX6700"
        "NX7700"
        "NX8700"
        "NX9700"
        "NX10700"
        "NX11700"
        "VRX2"
        "VRX4"
        "VRX6"
        "VRX8"
        "VXUNLICENSED"
        "VX500"
        "VX0000"
        "VX1000"
        "VX2000"
        "VX3000"
        "VX5000"
        "VX6000"
        "VX7000"
        "VX8000"
        "VX9000"
        "ECV"
        "ECXS"
        "EC10104"
        "EC10106"
        "EC10108"
        "EC10150"
        "EC10170"
        "ECUS"
        "ECS"
        "ECSP"
        "ECM"
        "ECMP"
        "ECMH"
        "ECMB"
        "ECL"
        "ECLP"
        "ECLB"
        "ECLH"
        "ECXL"
        "ECXLP"
        "ECXLB"
        "ECXLH"
        "ECXLH10G"
    )

    local model_supported=0
    for model in "${SUPPORTED_MODELS[@]}"; do
        if [ "${CURRENT_MODEL}" == "$model" ]; then
            model_supported=1
            break
        fi
    done

    if [ $model_supported -eq 0 ]; then
        echo "Model ${CURRENT_MODEL} is not supported by this image"
        return $COMPAT_FAIL_MODEL
    fi

    # Baseboards supported by this image
    SUPPORTED_BASEBOARDS=(
        "SILICOM_NA1000"
        "PER630"
        "PER640"
        "VMWARE"
        "VBOX"
        "XEN"
        "EC2"
        "AZURE"
        "AZURE_NVA"
        "GOOGLE"
        "ORACLE"
        "HYPERV"
        "KVM"
        "EQUINIX"
        "MEGAPORT"
        "ALIBABA_CLOUD"
        "LANNER_MB-7551W"
        "NC94510LF"
        "LANNER_NCB-1010W"
        "PER330"
        "PER340"
        "LANNER_MB-7584S"
        "0VWT90"
        "0DY523"
        "LANNER_MB7565S"
        "ADVANTECH_NAMB_SP1012MB"
        "ADVANTECH_NAMB_3050"
        "HPE_DL360"
        "HPE_DL325_GEN11"
        "LANNER_NCB-SP1513"
        "EC_10104"
        "EC_10106"
        "EC_10108"
    )

    local baseboard_supported=0
    for baseboard in "${SUPPORTED_BASEBOARDS[@]}"; do
        if [ "${CURRENT_BASEBOARD}" == "$baseboard" ]; then
            baseboard_supported=1
            break
        fi
    done

    if [ $baseboard_supported -eq 0 ]; then
        echo "Baseboard ${CURRENT_BASEBOARD} is not supported by this image"
        return $COMPAT_FAIL_BASEBOARD
    fi

    return $COMPAT_SUCCESS
}

# -----------------------------------------------------------------------------
# Set up the initramfs
#
# To correctly assemble Linux RAID devices we need a copy of mdadm.conf in the
# initramfs.
#
# This function requires its caller to change into the new image's boot
# directory, where initramfs.cpio.gz should be.
#
HAVE_UPGRADE_SETUP_INITRAMFS=y
upgrade_setup_initramfs()
{
    local tmpdir="/tmp/upgrade_setup_initramfs"
    local bootdir=`pwd`
    local initramfs=${bootdir}/initramfs.cpio.gz
    local initramfs_dist=${bootdir}/initramfs.dist.cpio.gz

    if [ -f /etc/mdadm.conf ] && [ -f ${initramfs} ]; then
        # Extract initramfs
        mkdir -p ${tmpdir}
        rm -rf ${tmpdir}/*
        cd ${tmpdir}
        /bin/gunzip -c ${initramfs} | /bin/cpio -idm

        # Add mdadm.conf to initramfs
        cp /etc/mdadm.conf etc/mdadm.conf

        # Re-package initramfs
        cp ${initramfs} ${initramfs_dist}
        find . | cpio --create --format=newc --quiet | /bin/gzip > ${initramfs}
        rm -rf ${tmpdir}
        cd ${bootdir}
    fi
}


# -----------------------------------------------------------------------------
# Build the contents of the /etc/fstab
#
# Modifies FSTAB_FILE, making any changes or additions required to boot the
# new image.
#
# FSTAB_FILE contains two sections.
#  1)  Mounts for partitions on physical disks, based on the layout settings
#      of the current model.
#  2)  Temporary and pseudo filesystem mounts; proc, devpts, tmpfs etc.
#
# We have writeimage.sh write the first section because it knows about layout
# settings. The second section is written by this function because it may be
# different per image.
#
# This function runs after writeimage.sh has created the FSTAB_FILE and written
# the first section, so it can modify the first section as well, if desired.
#
HAVE_UPGRADE_BUILD_FSTAB=y
upgrade_build_fstab()
{
    # If we're running FCx, we need to remove the last 5 lines writeimage.sh
    # added to FSTAB_FILE
    if [ $PLATFORM_VERSION -eq 0 ]; then
        sed -e :a -e '$d;N;2,5ba' -e 'P;D' -i ./${FSTAB_FILE}
        # Delete the swap partition mount, if present
        sed '/swap/d' -i ./${FSTAB_FILE}

        #
        # Convert the old disk naming scheme from the old one to
        # the new one.
        #
        sed 's/dev\/sda/dev\/sps\/slota/g' -i ./${FSTAB_FILE}
        sed 's/dev\/sdb/dev\/sps\/slotb/g' -i ./${FSTAB_FILE}
        sed 's/dev\/sdc/dev\/sps\/slotc/g' -i ./${FSTAB_FILE}
    fi

    # Write out the temp/pseudo filesystems required for this (Yocto) image
    cat >> ./${FSTAB_FILE} <<EOF
proc        /proc           proc    defaults        0 0
devpts      /dev/pts        devpts  gid=5,mode=620  0 0
tmpfs       /dev/shm        tmpfs   defaults        0 0
tmpfs       /run            tmpfs   mode=0755,nodev,nosuid,strictatime 0 0
tmpfs       /var/volatile   tmpfs   defaults        0 0
EOF
}

# -----------------------------------------------------------------------------
# Upgrade hooks for FC6 and earlier.
#
# Instead of modifying writeimage.sh in all FCx branches to find and run the
# hooks in this upgrade script, we use a hook in this image's build_version.sh
# to trick the old FCx writeimage.sh into inserting hooks on the fly.
#
upgrade_fc6_writeimage_hooks()
{
    if [ $PLATFORM_VERSION -eq 0 ]; then

        upgrade_init

        # ---------------------------------------------------------------------
        # Rename a function using declare
        #
        # (based on http://mivok.net/2009/09/20/bashfunctionoverrist.html)
        #
        rename_function()
        {
            # Capture the old function
            local ORIG_FUNC=$(declare -f $1)
            [ -z "$ORIG_FUNC" ] && local ORIG_FUNC="$1 () { return; }"

            # Define old function under the new name
            local NEWNAME_FUNC="$2${ORIG_FUNC#$1}"
            eval "$NEWNAME_FUNC"
        }

        # ---------------------------------------------------------------------
        # Run the compatibility checks from graft point #4.
        #
        HAVE_WRITEIMAGE_GRAFT_4=y
        rename_function writeimage_graft_4 writeimage_graft_4_original
        writeimage_graft_4()
        {
            upgrade_check_compatibility
            local rc=$?
            if [ ! $rc -eq 0 ]; then
                cleanup
                exit $rc
            fi

            # Run the original graft function
            writeimage_graft_4_original
        }

        # ---------------------------------------------------------------------
        # Modify FSTAB_FILE & setup initramfs from graft point #3.
        #
        HAVE_WRITEIMAGE_GRAFT_3=y
        rename_function writeimage_graft_3 writeimage_graft_3_original
        writeimage_graft_3()
        {
            # Mount the new image's boot directory and change into it so that
            # we can set up the initramfs
            if [ -z "${LAST_BOOT_MNT}" ]; then
                echo "*** Could not determine last boot mount"
                cleanup_exit
            fi
            local tmp_mnt="/mnt/tmp_root_upgrade_sh"
            mkdir -p ${tmp_mnt}
            mount ${LAST_BOOT_MNT} ${tmp_mnt}
            if [ $? -eq 1 ]; then
                echo "*** Could not mount ${LAST_BOOT_MNT} on ${tmp_mnt}"
                cleanup_exit
            fi

            # Setup the initramfs
            local olddir=`pwd`
            cd ${tmp_mnt}
            upgrade_setup_initramfs
            cd ${olddir}

            # Unmount the boot directory
            umount ${tmp_mnt}
            rm -rf  ${tmp_mnt}

            upgrade_build_fstab

            #
            # Move the block clean and the FTW active file from old location
            # to the new location.
            #
            if [ -f /var/tmp/block_clean.txt ] ; then
                touch /var/opt/tms/block_clean.conf
            fi

            if [ -f /var/tmp/ftw_active.txt ] ; then
                touch /var/opt/tms/ftw_active.conf
            fi

            # Make sure /run/lock/subsys and /var/volatile exist in the running
            # FCx image so that if we fallback from Yocto, the new symlinks
            # into Yocto's volatile tmpfs will point to the old FCx locations.
            mkdir -p /run/lock/subsys
            mkdir -p /var/volatile/tmp
            
            # Run the original graft function
            writeimage_graft_3_original
        }
    fi
}
